% Program to plot the Dow Jones Industrial Average from 1928 to May 2025, and compute the standard deviation.
% Data available for download at https://finance.yahoo.com/quote/%5EDJI/history?p=%5EDJI
% Just set the Time Period, then find and click the download link, but you ned a paid version of Yahoo.
%
% If you have a subscription for Microsoft Office 365, you can also get historical stock prices.
% Reference: https://support.microsoft.com/en-us/office/stockhistory-function-1ac8b5b3-5f62-4d94-8ab8-7504ec7239a8#:~:text=The%20STOCKHISTORY%20function%20retrieves%20historical,Microsoft%20365%20Business%20Premium%20subscription.
% For example put this in an Excel Cell
% =STOCKHISTORY("^DJI", "1/1/2000", "5/10/2025", 0, 1, 0, 1,2,3,4, 5)
% and it will fill out a table in Excel
%====================================================================================================================
clc;    % Clear the command window.
close all;  % Close all figures (except those of imtool.)
imtool close all;  % Close all imtool figures if you have the Image Processing Toolbox.
clear;  % Erase all existing variables. Or clearvars if you want.
workspace;  % Make sure the workspace panel is showing.
format long g;
format compact;
fontSize = 14;

filename = 'Dow Jones Industrial Index.xlsx';
data = readtable(filename);
% Date,Close,Open,High,Low,Volume
dates = data.Date;
closing = data.Close;
volume = data.Volume;

% Define start date and stop date
startDate = datetime(2011,1,1)
stopDate = dates(end)
selectedDates = dates > startDate;

% Extract those dates:
dates = dates(selectedDates);
closing = closing(selectedDates);
volume = volume(selectedDates);

% Plot Volume
hFigVolume = figure('Name', 'Daily Volume');
plot(dates, volume, 'b-');
grid on;
xticks(startDate:calendarDuration(5,0,0):stopDate)
title('Dow Jones Industrial Average Volume', 'FontSize', fontSize);

hFig = figure('Name', 'Daily Standard Deviation');
subplot(3, 1, 1);
plot(dates, closing, 'b-');
xticks(startDate:calendarDuration(5,0,0):stopDate)
drawnow;
grid on;
caption = sprintf('Dow Jones Industrial Average from %s through %s', dates(1), dates(end));
title(caption, 'FontSize', fontSize);

% Get the average change from one trading day to the next.
diffs = 100 * abs(closing(2:end) - closing(1:end-1)) ./ closing(1:end-1);
subplot(3, 1, 2);
averageDailyChange = mean(diffs)
% Looks pretty noisy so let's smooth it for a nicer display.
numWeeks = 4;
diffs = sgolayfilt(diffs, 2, 5*numWeeks+1);
plot(dates(2:end), diffs, 'b-');
grid on;
xticks(startDate:calendarDuration(5,0,0):stopDate)
hold on;
line(xlim, [averageDailyChange, averageDailyChange], 'Color', 'r', 'LineWidth', 2);
ylabel('Percentage', 'FontSize', fontSize);
caption = sprintf('Day-to-Day Change Percentage.  Average Daily Change (from prior day) = %.2f%%', averageDailyChange);
title(caption, 'FontSize', fontSize);
drawnow;

% Get the stddev over a 5 trading day window.
sd = stdfilt(closing, ones(5, 1));
% Get it relative to the magnitude.
sd = sd ./ closing * 100;
averageVariation = mean(sd)
numWeeks = 2;
% Looks pretty noisy so let's smooth it for a nicer display.
sd = sgolayfilt(sd, 2, 5*numWeeks+1);
% Plot it.
subplot(3, 1, 3);
plot(dates, sd, 'b-');
grid on;
xticks(startDate:calendarDuration(5,0,0):stopDate)
hold on;
line(xlim, [averageVariation, averageVariation], 'Color', 'r', 'LineWidth', 2);
ylabel('Percentage', 'FontSize', fontSize);
caption = sprintf('Weekly Standard Deviation, Averaged Over %d Weeks (%d trading days).  Mean SD = %.2f', ...
	numWeeks, 5*numWeeks+1, averageVariation);
title(caption, 'FontSize', fontSize);
% Maximize figure window.
g = gcf;
g.WindowState = 'maximized';
